/**
* \file: mlink_wl.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* MLINK WL Adapter
*
* \component: mlink
*
* \author: Michael Methner ADITG/SW1 mmethner@de.adit-jv.com
*
* \copyright: (c) 2003 - 2013 ADIT Corporation
*
* \history
* 0.1 Michael Methner Initial version
*
***********************************************************************/


#include <poll.h>
#include <pthread.h>
#include <linux/input.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <compositor-shim.h>
#include <wayland-client.h>
#include <wayland-egl.h>

#include <mlink_wl.h>
#include <mlink_wl_if.h>
#include <mlink_wl_internals.h>

#include "vcdfcl.h"

#include "vnccommondecoderapplicationcontextimpl.h"

/* PRQA: Lint Message 160, 505, 578: deactivation because DLT macros generate this lint findings*/
/*lint -e160 -e505 -e578*/

/* PRQA: Lint Message 40, 413, 613: deactivation because OSS(Wayland) macros generate these lint findings*/
/*lint -e40 -e413 -e613*/

/* PRQA: Lint Message 429: deactivation because pWlSeat is freed in RemoveSeat*/
/*lint -e429*/

VNCCommonDecoderApplicationContext context_from_main_application_to_plugin_renderer;
pthread_t waitWayland;

// Lock protects access to wl_ctx - to avoid deletion while it is in use
static int g_initialized_rwlock = 0;
static pthread_rwlock_t g_rwlock;


void
mlink_free_mem(void** input_ptr)
{
    if(*input_ptr != NULL)
    {
        free(*input_ptr);
        *input_ptr = NULL;
    }
}

static void
send_touch(mlink_wl_context * p_wl_ctx)
{
  VNCTouchDescriptor * touch_ptr=NULL;
  int size=0;

  if(p_wl_ctx->touch_valid[0])
    {
      touch_ptr= &p_wl_ctx->touch_desc[0];
      if(p_wl_ctx->touch_valid[1])
        {
          size = 2;
        }
      else
        {
          size = 1;
        }
    }
  else if(p_wl_ctx->touch_valid[1])
    {
      touch_ptr= &p_wl_ctx->touch_desc[1];
      size = 1;
    }
  if(p_wl_ctx->p_dlt_ctx)
  {
      DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
              DLT_STRING("send_touch: Touch Event sent via SDK"));
  }


  (p_wl_ctx->pViewerSdk->vncViewerSendTouchEvent)(p_wl_ctx->pViewer,
      touch_ptr, size, 0, 0);
  if(p_wl_ctx->p_dlt_ctx)
  {
      DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
              DLT_STRING("send_touch: After having returned from SDK"));
  }
}

static void
TouchHandleDown(void* data, struct wl_touch* touch, uint32_t serial, uint32_t wltime,
                struct wl_surface* surface, int32_t id, wl_fixed_t xw, wl_fixed_t yw)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(touch);
    MLINK_UNUSED(serial);
    MLINK_UNUSED(wltime);
    MLINK_UNUSED(surface);
    MLINK_UNUSED(id);
    MLINK_UNUSED(xw);
    MLINK_UNUSED(yw);
    WlSeat * pWlSeat = (WlSeat *)data;
    mlink_wl_context * p_wl_ctx = pWlSeat->p_wl_context;
    int pointer_x = (int)wl_fixed_to_double(xw);
    int pointer_y = (int)wl_fixed_to_double(yw);

    if (p_wl_ctx->support_touch)
    {
        if(id>=0 && id<MLINK_WL_MAX_TOUCHPOINTS)
          {
            p_wl_ctx->touch_desc[id].location.x = pointer_x;
            p_wl_ctx->touch_desc[id].location.y = pointer_y;
            p_wl_ctx->touch_desc[id].pressure = 1;
            p_wl_ctx->touch_desc[id].buttonMask = VNCPointerDeviceButtonLeft;
            p_wl_ctx->touch_desc[id].identifier = id;
            p_wl_ctx->touch_valid[id] = 1;
            if(p_wl_ctx->p_dlt_ctx)
            {
                DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                        DLT_STRING("TouchHandleDown"));
            }
            send_touch(p_wl_ctx);
          }
    }
    else
    {
        if(id==0)
          {
            (p_wl_ctx->pViewerSdk->vncViewerSendPointerEvent)(p_wl_ctx->pViewer,
                VNCPointerDeviceButton1, (vnc_uint16_t) pointer_x,
                (vnc_uint16_t) pointer_y);
          }
    }

    p_wl_ctx->last_pointer_x = pointer_x;
    p_wl_ctx->last_pointer_y = pointer_y;
}


static void
TouchHandleUp(void* data, struct wl_touch* touch, uint32_t serial, uint32_t wltime, int32_t id)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(touch);
    MLINK_UNUSED(serial);
    MLINK_UNUSED(wltime);
    MLINK_UNUSED(id);

    WlSeat * pWlSeat = (WlSeat *)data;
    mlink_wl_context * p_wl_ctx = pWlSeat->p_wl_context;

    if (p_wl_ctx->support_touch)
    {
        if(id>=0 && id<MLINK_WL_MAX_TOUCHPOINTS)
          {
            p_wl_ctx->touch_desc[id].pressure = 0;
            p_wl_ctx->touch_desc[id].buttonMask = VNCPointerDeviceButtonLeft;
            p_wl_ctx->touch_desc[id].identifier = id;
            if(p_wl_ctx->p_dlt_ctx)
            {
                DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                        DLT_STRING("TouchHandleUp"));
            }
            send_touch(p_wl_ctx);
            p_wl_ctx->touch_valid[id] = 0;
          }
    }
  else
    {
      if(id==0)
        {
          (p_wl_ctx->pViewerSdk->vncViewerSendPointerEvent)(p_wl_ctx->pViewer,
              VNCPointerDeviceButtonNone, (vnc_uint16_t) p_wl_ctx->last_pointer_x,
              (vnc_uint16_t) p_wl_ctx->last_pointer_y);
        }
    }
}

static void
TouchHandleMotion(void* data, struct wl_touch* touch, uint32_t wltime, int32_t id,
                  wl_fixed_t xw, wl_fixed_t yw)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(touch);
    MLINK_UNUSED(wltime);
    MLINK_UNUSED(id);

    int pointer_x = (int)wl_fixed_to_double(xw);
    int pointer_y = (int)wl_fixed_to_double(yw);

    WlSeat * pWlSeat = (WlSeat *)data;
    mlink_wl_context * p_wl_ctx = pWlSeat->p_wl_context;

  if (p_wl_ctx->support_touch)
    {
      if(id>=0 && id<MLINK_WL_MAX_TOUCHPOINTS)
        {
          p_wl_ctx->touch_desc[id].location.x = pointer_x;
          p_wl_ctx->touch_desc[id].location.y = pointer_y;
          p_wl_ctx->touch_desc[id].pressure = 1;
          p_wl_ctx->touch_desc[id].buttonMask = VNCPointerDeviceButtonLeft;
          p_wl_ctx->touch_desc[id].identifier = id;
          p_wl_ctx->touch_valid[id] = 1;
          if(p_wl_ctx->p_dlt_ctx)
          {
              DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                      DLT_STRING("TouchHandleMotion"));
          }
          send_touch(p_wl_ctx);
        }
    }
  else
    {
      if(id==0)
        {
          (p_wl_ctx->pViewerSdk->vncViewerSendPointerEvent)(p_wl_ctx->pViewer,
              VNCPointerDeviceButton1, (vnc_uint16_t) pointer_x,
              (vnc_uint16_t) pointer_y);
          p_wl_ctx->last_pointer_x = pointer_x;
          p_wl_ctx->last_pointer_y = pointer_y;
        }
    }
}


static void
TouchHandleFrame(void* data, struct wl_touch* touch)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(touch);
}


static void
TouchHandleCancel(void* data, struct wl_touch* touch)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(touch);
}



static const struct wl_touch_listener TouchListener = {
    TouchHandleDown,
    TouchHandleUp,
    TouchHandleMotion,
    TouchHandleFrame,
    TouchHandleCancel,
    NULL,
    NULL
};


void
PointerHandleEnter(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                   struct wl_surface* wlSurface, wl_fixed_t sx, wl_fixed_t sy)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(wlPointer);
    MLINK_UNUSED(serial);
    MLINK_UNUSED(wlSurface);
    MLINK_UNUSED(sx);
    MLINK_UNUSED(sy);
}

void
PointerHandleLeave(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                   struct wl_surface* wlSurface)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(wlPointer);
    MLINK_UNUSED(serial);
    MLINK_UNUSED(wlSurface);
}

void
PointerHandleMotion(void* data, struct wl_pointer* wlPointer, uint32_t time,
                    wl_fixed_t sx, wl_fixed_t sy)
{
    MLINK_UNUSED(wlPointer);
    MLINK_UNUSED(time);
    vnc_uint16_t pointer_x = (vnc_uint16_t)wl_fixed_to_double(sx);
    vnc_uint16_t pointer_y = (vnc_uint16_t)wl_fixed_to_double(sy);

    WlSeat * pWlSeat = (WlSeat *)data;
    mlink_wl_context * p_wl_ctx = pWlSeat->p_wl_context;

    (p_wl_ctx->pViewerSdk->vncViewerSendPointerEvent)(
        p_wl_ctx->pViewer, p_wl_ctx->pointer_button_state,
        pointer_x, pointer_y);

    p_wl_ctx->pointer_x = pointer_x;
    p_wl_ctx->pointer_y = pointer_y;
}

void
PointerHandleButton(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                    uint32_t time, uint32_t button, uint32_t state)
{
    MLINK_UNUSED(wlPointer);
    MLINK_UNUSED(serial);
    MLINK_UNUSED(time);

    WlSeat * pWlSeat = (WlSeat *)data;
    mlink_wl_context * p_wl_ctx = pWlSeat->p_wl_context;

    VNCPointerDeviceButton vnc_button;
    switch(button)
    {
    case BTN_TOUCH:
      vnc_button = VNCPointerDeviceButtonLeft;
      break;
    case BTN_LEFT:
      vnc_button = VNCPointerDeviceButtonLeft;
      break;
    case BTN_RIGHT:
      vnc_button = VNCPointerDeviceButtonRight;
      break;
    case BTN_MIDDLE:
      vnc_button = VNCPointerDeviceButtonMiddle;
      break;
    default:
      vnc_button = VNCPointerDeviceButtonNone;
    }

    if(vnc_button != VNCPointerDeviceButtonNone)
      {
        if(state==0)
          {
            p_wl_ctx->pointer_button_state =
                p_wl_ctx->pointer_button_state & (VNCPointerDeviceButton)~vnc_button;
          }
        else
          {
            p_wl_ctx->pointer_button_state =
                p_wl_ctx->pointer_button_state | vnc_button;
          }
      }

    if(p_wl_ctx->pointer_button_state == VNCPointerDeviceButtonNone)
      {
        (p_wl_ctx->pViewerSdk->vncViewerSendPointerEvent)(
            p_wl_ctx->pViewer, p_wl_ctx->pointer_button_state,
            p_wl_ctx->pointer_x, p_wl_ctx->pointer_y);
      }
}


void
PointerHandleAxis(void* data, struct wl_pointer* wlPointer, uint32_t time,
                  uint32_t axis, wl_fixed_t value)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(wlPointer);
    MLINK_UNUSED(time);
    MLINK_UNUSED(axis);
    MLINK_UNUSED(value);
}

void PointerHandleFrame(void *data, struct wl_pointer *pointer)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(pointer);
}

void PointerHandleAxisSource(void *data, struct wl_pointer *pointer, uint32_t source)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(pointer);
    MLINK_UNUSED(source);
}

void PointerHandleAxisStop(void *data, struct wl_pointer *pointer,
        uint32_t time, uint32_t axis)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(pointer);
    MLINK_UNUSED(time);
    MLINK_UNUSED(axis);
}

void PointerHandleAxisDiscrete(void *data, struct wl_pointer *pointer,
        uint32_t axis, int32_t discrete)
{
    MLINK_UNUSED(data);
    MLINK_UNUSED(pointer);
    MLINK_UNUSED(axis);
    MLINK_UNUSED(discrete);
}

const struct wl_pointer_listener PointerListener = {
    PointerHandleEnter,
    PointerHandleLeave,
    PointerHandleMotion,
    PointerHandleButton,
    PointerHandleAxis,
    PointerHandleFrame,
    PointerHandleAxisSource,
    PointerHandleAxisStop,
    PointerHandleAxisDiscrete
};

static void
SeatHandleCapabilities(void* data, struct wl_seat* seat, uint32_t caps)
{
    MLINK_UNUSED(seat);
    WlSeat * pWlSeat = (WlSeat *)data;
    mlink_wl_context * p_wl_ctx = (mlink_wl_context *)pWlSeat->p_wl_context;
    struct wl_seat* wlSeat = pWlSeat->p_wl_seat;
    if (!wlSeat)
        return;

    if (p_wl_ctx->flags & MLINK_WL_ENABLE_TOUCHEVENTS)
    {
        if ((caps & WL_SEAT_CAPABILITY_TOUCH) && !pWlSeat->p_wl_touch)
        {
            pWlSeat->p_wl_touch = wl_seat_get_touch(wlSeat);
            wl_touch_set_user_data(pWlSeat->p_wl_touch, pWlSeat);
            wl_touch_add_listener(pWlSeat->p_wl_touch, &TouchListener, pWlSeat);
        }
        else if (!(caps & WL_SEAT_CAPABILITY_TOUCH) && pWlSeat->p_wl_touch)
        {
            wl_touch_destroy(pWlSeat->p_wl_touch);
            pWlSeat->p_wl_touch = NULL;
        }

        if ((caps & WL_SEAT_CAPABILITY_POINTER) && !pWlSeat->p_wl_pointer)
        {
        	pWlSeat->p_wl_pointer = wl_seat_get_pointer(wlSeat);
            wl_pointer_set_user_data(pWlSeat->p_wl_pointer, pWlSeat);
            wl_pointer_add_listener(pWlSeat->p_wl_pointer, &PointerListener, pWlSeat);
        }
        else if (!(caps & WL_SEAT_CAPABILITY_POINTER) && pWlSeat->p_wl_pointer)
        {
            wl_pointer_destroy(pWlSeat->p_wl_pointer);
            pWlSeat->p_wl_pointer = NULL;
        }

    }
}

static void
SeatHandleName(void* data, struct wl_seat* seat, const char* name)
{
    MLINK_UNUSED(seat);
    WlSeat * pWlSeat = (WlSeat *)data;
    pWlSeat->p_seat_name = strdup(name);
}

static struct wl_seat_listener seatListener = {
    SeatHandleCapabilities,
    SeatHandleName
};

static void
AddSeat(struct mlink_wl_context *p_wl_ctx, uint32_t name)
{
    struct WlSeat *pWlSeat;

    pWlSeat = malloc(sizeof *pWlSeat);
    if(pWlSeat == NULL)
    {
        if(p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_ERROR,
                    DLT_STRING("mlink_wl: Not able to allocate memory for seat id:"),
                    DLT_UINT32(name));
        }
        return;
    }
    pWlSeat->p_wl_touch = NULL;
    pWlSeat->p_wl_pointer = NULL;
    pWlSeat->p_seat_name = NULL;
    pWlSeat->seat_id = name;
    pWlSeat->p_wl_context = p_wl_ctx;
    pWlSeat->p_wl_seat = wl_registry_bind(p_wl_ctx->p_wl_registry, name,
                                      &wl_seat_interface, 5);
    wl_seat_add_listener(pWlSeat->p_wl_seat, &seatListener, pWlSeat);
    wl_proxy_set_queue((struct wl_proxy*) pWlSeat->p_wl_seat, p_wl_ctx->p_wl_input_queue);
    pthread_mutex_lock(&p_wl_ctx->lock_seat);
    wl_list_insert(&p_wl_ctx->seats, &pWlSeat->link);
    pthread_mutex_unlock(&p_wl_ctx->lock_seat);
}

static void
RemoveSeat(struct WlSeat *pWlSeat)
{
    if(pWlSeat)
    {
        if (pWlSeat->p_seat_name)
        {
            free(pWlSeat->p_seat_name);
            pWlSeat->p_seat_name = NULL;
        }
        if (pWlSeat->p_wl_touch)
        {
            wl_touch_destroy(pWlSeat->p_wl_touch);
            pWlSeat->p_wl_touch = NULL;
        }
        if (pWlSeat->p_wl_pointer)
        {
            wl_pointer_destroy(pWlSeat->p_wl_pointer);
            pWlSeat->p_wl_pointer = NULL;
        }
        if (pWlSeat->p_wl_seat)
        {
            wl_seat_destroy(pWlSeat->p_wl_seat);
            pWlSeat->p_wl_seat = NULL;
        }
        wl_list_remove(&pWlSeat->link);
        free(pWlSeat);
        pWlSeat = NULL;
    }
}

static void RegistryHandleGlobal(void* data, struct wl_registry* registry,
        uint32_t name, const char* interface, uint32_t version)
{
    mlink_wl_context * p_wl_ctx = (mlink_wl_context *) data;

    mlink_wl_if_init_global_handler(data, registry, name, interface, version);


    if (!p_wl_ctx->is_h264_enabled)
    {
        if (!strcmp(interface, "wl_compositor"))
        {
            p_wl_ctx->p_wl_compositor = (struct wl_compositor*) (wl_registry_bind(
                    registry, name, &wl_compositor_interface, 1));
        }
    }

    if (!strcmp(interface, "wl_seat"))
    {
        AddSeat(p_wl_ctx, name);
    }
}

static void HandleGlobalRemove (void *data, struct wl_registry *registry, uint32_t name)
{
    MLINK_UNUSED(registry);
    mlink_wl_context * p_wl_ctx = (mlink_wl_context *) data;

    pthread_mutex_lock(&p_wl_ctx->lock_seat);
    WlSeat *pWlSeat = NULL;
    WlSeat *pWlSeat_tmp = NULL;
    wl_list_for_each_safe(pWlSeat, pWlSeat_tmp, &p_wl_ctx->seats, link)
    {
        if (pWlSeat->seat_id == name)
        {
            RemoveSeat(pWlSeat);
            break;
        }
    }
    pthread_mutex_unlock(&p_wl_ctx->lock_seat);

}

static struct wl_registry_listener registryListener =
{ RegistryHandleGlobal, HandleGlobalRemove };


static int mlink_wl_create_image_buffer(mlink_wl_context * p_wl_ctx, int width,
        int height);

int mlink_connect_to_wayland(mlink_wl_context * p_wl_ctx)
{
    if (p_wl_ctx->p_wl_display == NULL)
    {
        if(p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("mlink_connect_to_wayland: wl_display_connect failed"));
        }
        return 0;
    }

    wl_list_init(&p_wl_ctx->seats);
    pthread_mutex_init(&p_wl_ctx->lock_seat, NULL);

    p_wl_ctx->p_wl_input_queue = wl_display_create_queue(p_wl_ctx->p_wl_display);
    if (NULL == p_wl_ctx->p_wl_input_queue)
    {
      if (p_wl_ctx->p_dlt_ctx)
        {
          DLT_LOG(*p_wl_ctx->p_dlt_ctx, DLT_LOG_INFO,
              DLT_STRING("mlink_connect_to_wayland: wl_input_queue not set"));
        }

      return 0;
    }

    p_wl_ctx->p_wl_registry = wl_display_get_registry(p_wl_ctx->p_wl_display);
    wl_registry_add_listener(p_wl_ctx->p_wl_registry, &registryListener, p_wl_ctx);
    wl_display_dispatch(p_wl_ctx->p_wl_display);

    wl_display_roundtrip(p_wl_ctx->p_wl_display);

    if (!p_wl_ctx->is_h264_enabled)
    {
        p_wl_ctx->wlAdapterContext = compositor_shim_initialize(p_wl_ctx->p_wl_display);

        if(p_wl_ctx->wlAdapterContext == NULL)
        {
            if(p_wl_ctx->p_dlt_ctx)
            {
                DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                        DLT_STRING("mlink_connect_to_wayland: compositor_shim_initialize failed"));
            }

            return 0;
        }

        if (p_wl_ctx->p_wl_compositor == NULL)
        {
            if(p_wl_ctx->p_dlt_ctx)
            {
                DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                        DLT_STRING("mlink_wl: wl_compositor not set"));
            }

            return 0;
        }
    }
    else
    {
        p_wl_ctx->intialized = 1;
    }

    return 1;
}

void *wait_for_wayland(void *arg)
{
    mlink_wl_context * p_wl_ctx = (mlink_wl_context *)arg;
    while (1)
    {
        usleep(100000);
        if(p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("wait_for_wayland"));
        }
        if (context_from_main_application_to_plugin_renderer->container.is_display_ready == 1
            || context_from_main_application_to_plugin_renderer->container.is_streaming_stopped == 1)
            break;
    }
    if (context_from_main_application_to_plugin_renderer->container.is_streaming_stopped == 1)
    {
        if(p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("wait_for_wayland is_streaming_stopped == 1"));
        }
    }
    else
    {
        p_wl_ctx->p_wl_display = (struct wl_display*)context_from_main_application_to_plugin_renderer->container.p_wayland;
        if(p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("wait_for_wayland: p_wl_ctx->p_wl_display address"), DLT_PTR(p_wl_ctx->p_wl_display));
        }
        mlink_connect_to_wayland(p_wl_ctx);
    }
    return NULL;
}

int mlink_wl_create_wl_context(mlink_wl_context * p_wl_ctx)
{
    if (p_wl_ctx->is_h264_enabled)
    {
        if(p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("mlink_wl_create_wl_context: This is H.264, change flow!!! Wait for wayland context and return 1 here"));
        }
        pthread_create(&waitWayland, 0, wait_for_wayland, p_wl_ctx);

        return 1;
    }
    else
    {
        p_wl_ctx->p_wl_display = wl_display_connect(NULL);
    }

    return mlink_connect_to_wayland(p_wl_ctx);

}


int mlink_wl_create_surface(mlink_wl_context * p_wl_ctx, int width, int height)
{
    int error = 0;
    int scr_width;
    int scr_height;
    int scr_x_offset;
    int scr_y_offset;


    error = compositor_shim_surface_init(&p_wl_ctx->wlAdapterSurfaceContext,
            p_wl_ctx->p_wl_surface, p_wl_ctx->layer_id, p_wl_ctx->surface_id,
            width, height);
    if(error != 0 && p_wl_ctx->p_dlt_ctx)
    {
        DLT_LOG(*p_wl_ctx->p_dlt_ctx, DLT_LOG_ERROR,
                DLT_STRING("mlink_wl: compositor_shim_surface_init failed with error code: "),
                DLT_UINT32(error));
        return error;
    }

    if(p_wl_ctx->force_fullscreen)
    {
    	scr_width = p_wl_ctx->screen_width;
    	scr_height = p_wl_ctx->screen_height;
    }
    else
    {
    	float screen_aspect = (float)p_wl_ctx->screen_width / p_wl_ctx->screen_height;
    	float phone_aspect = (float)width / height;

    	if(screen_aspect < phone_aspect)
    	{
    		scr_width = p_wl_ctx->screen_width;
    		scr_height = (int)((float)p_wl_ctx->screen_width / phone_aspect);
    	}
    	else
    	{
    		scr_height = p_wl_ctx->screen_height;
    		scr_width = (int)((float)p_wl_ctx->screen_height * phone_aspect);
    	}
    }

    scr_x_offset = (p_wl_ctx->screen_width-scr_width)/2 + p_wl_ctx->screen_x_offset;
    scr_y_offset = (p_wl_ctx->screen_height-scr_height)/2 + p_wl_ctx->screen_y_offset;

    DLT_LOG(*p_wl_ctx->p_dlt_ctx, DLT_LOG_INFO, DLT_STRING("mlink_wl: Destination Surface"),
    		DLT_STRING("destWidth="),DLT_INT32(scr_width),
    		DLT_STRING("destHeight="),DLT_INT32(scr_height),
    		DLT_STRING("destX="),DLT_INT32(scr_x_offset),
    		DLT_STRING("destY="),DLT_INT32(scr_y_offset) );

    p_wl_ctx->wlAdapterSurfaceContext.destWidth = scr_width;
    p_wl_ctx->wlAdapterSurfaceContext.destHeight = scr_height;

    p_wl_ctx->wlAdapterSurfaceContext.destX = scr_x_offset;
    p_wl_ctx->wlAdapterSurfaceContext.destY = scr_y_offset;


    error = compositor_shim_surface_configure(p_wl_ctx->wlAdapterContext, &p_wl_ctx->wlAdapterSurfaceContext,
            ADAPTER_CONFIGURATION_ALL);
    if(error != 0 && p_wl_ctx->p_dlt_ctx)
    {
        DLT_LOG(*p_wl_ctx->p_dlt_ctx, DLT_LOG_ERROR,
                DLT_STRING("mlink_wl: compositor_shim_surface_configure failed with error code: "),
                DLT_UINT32(error));
        return error;
    }
    return error;
}


static void mlink_wl_destroy_wl_context(mlink_wl_context * p_wl_ctx)
{

    p_wl_ctx->intialized = 0;

    if (!p_wl_ctx->is_h264_enabled)
    {
        if (p_wl_ctx->wlAdapterContext)
        {
            compositor_shim_terminate(p_wl_ctx->wlAdapterContext);
            p_wl_ctx->wlAdapterContext = NULL;
        }

        pthread_mutex_lock(&p_wl_ctx->lock_seat);
        WlSeat *pWlSeat = NULL;
        WlSeat *pWlSeat_tmp = NULL;
        wl_list_for_each_safe(pWlSeat,pWlSeat_tmp, &p_wl_ctx->seats, link)
        {
            RemoveSeat(pWlSeat);
        }
        pthread_mutex_unlock(&p_wl_ctx->lock_seat);

        if (p_wl_ctx->p_wl_compositor)
        {
            wl_compositor_destroy(p_wl_ctx->p_wl_compositor);
            p_wl_ctx->p_wl_compositor = NULL;
        }

        if (p_wl_ctx->p_wl_registry)
        {
            wl_registry_destroy(p_wl_ctx->p_wl_registry);
            p_wl_ctx->p_wl_registry = NULL;
        }

        if (p_wl_ctx->p_wl_input_queue)
        {
            wl_event_queue_destroy(p_wl_ctx->p_wl_input_queue);
            p_wl_ctx->p_wl_input_queue = NULL;
        }

        if(p_wl_ctx->p_wl_display)
        {
            wl_display_disconnect(p_wl_ctx->p_wl_display);
            p_wl_ctx->p_wl_display = NULL;
        }
    }
    pthread_mutex_destroy(&p_wl_ctx->lock_seat);

}

static int mlink_wl_create_image_buffer(mlink_wl_context * p_wl_ctx, int width,
        int height)
{
    if (p_wl_ctx->is_h264_enabled && p_wl_ctx->p_wl_display != NULL)
    {
        p_wl_ctx->intialized = 1;
    }
    else
    {
        p_wl_ctx->p_wl_surface
                = wl_compositor_create_surface(p_wl_ctx->p_wl_compositor);
        if(p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("mlink_wl: created p_wl_ctx->wl_surface: "), DLT_PTR(p_wl_ctx->p_wl_surface));
        }

        if (p_wl_ctx->p_wl_surface == NULL)
        {
            if(p_wl_ctx->p_dlt_ctx)
            {
                DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_ERROR,
                        DLT_STRING("mlink_wl: wl_compositor_create_surface failed"));
            }
            return 0;
        }

        int error = mlink_wl_create_surface(p_wl_ctx, width, height);
        if (error != 0)
        {
            if(p_wl_ctx->p_dlt_ctx)
            {
                DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_ERROR,
                        DLT_STRING("mlink_wl: creating surface failed"), DLT_INT32(error));
            }
            return 0;
        }

        if(mlink_wl_if_create_img(p_wl_ctx, width, height))
        {
            p_wl_ctx->intialized = 1;
        }
        else
        {
            p_wl_ctx->intialized = 0;
        }
    }

    return p_wl_ctx->intialized;
}




static void mlink_wl_free_image_buffer(mlink_wl_context * p_wl_ctx)
{
    if (p_wl_ctx->is_h264_enabled)
    {
        p_wl_ctx->intialized = 0;
    }
    else
    {
        int error = compositor_shim_surface_destroy(&p_wl_ctx->wlAdapterSurfaceContext);
        if(error != 0 && p_wl_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_wl_ctx->p_dlt_ctx, DLT_LOG_ERROR,
                    DLT_STRING("mlink_wl: compositor_shim_surface_destroy failed with error code: "),
                    DLT_INT32(error));
        }

        mlink_wl_if_destroy_img(p_wl_ctx);
        p_wl_ctx->intialized = 0;

        if (p_wl_ctx->p_wl_surface)
        {
            wl_surface_destroy(p_wl_ctx->p_wl_surface);
            p_wl_ctx->p_wl_surface = NULL;
        }
    }
}


mlink_wl_context * mlink_wl_initialize(VNCViewerSDK * pViewerSdk,
        VNCViewer *pViewer, DltContext * p_dlt_ctx,
        unsigned int layer_id, unsigned int surface_id,
        VNCPixelFormat * p_framebuffer_pixelformat,
        int screen_x_offset, int screen_y_offset,
        int screen_width, int screen_height, unsigned int force_fullscreen,
        enum mlink_wl_flags flags,
        mlink_wl_error_callback * pErrorCb,
        int is_h264_enabled)
{
    if(p_dlt_ctx)
    {
        DLT_LOG(*p_dlt_ctx,DLT_LOG_INFO,
                DLT_STRING("mlink_wl_initialize: version:"), DLT_STRING(COMP_GIT_VERSION));
    }
    if (is_h264_enabled == 1)
    {
        if(p_dlt_ctx)
        {
            DLT_LOG(*p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("mlink_wl_initialize: H264 enabled"));
        }
        // Load Plugin decoder
        const VCDFCLLoadPluginFromLibraryDetails gstreamerDecoder = {NULL, "gstreamer"};
        VCDFCLInitDetails initDetail;
        memset(&initDetail, 0, sizeof(initDetail));
        initDetail.structSize = sizeof(initDetail);
        initDetail.loadFromLibraryDetails = &gstreamerDecoder;
        initDetail.maxInFlightPayloads = 1;

        context_from_main_application_to_plugin_renderer = malloc(sizeof(VNCCommonDecoderApplicationContextImpl));
        memset(context_from_main_application_to_plugin_renderer, 0, sizeof(VNCCommonDecoderApplicationContextImpl));
        context_from_main_application_to_plugin_renderer->container.layer_id = layer_id;
        context_from_main_application_to_plugin_renderer->container.surface_id = surface_id;
        context_from_main_application_to_plugin_renderer->container.is_display_ready = -1;
        context_from_main_application_to_plugin_renderer->container.is_streaming_stopped = 0;
        context_from_main_application_to_plugin_renderer->container.screen_height = screen_height;
        context_from_main_application_to_plugin_renderer->container.screen_width = screen_width;
        context_from_main_application_to_plugin_renderer->container.screen_x_offset = screen_x_offset;
        context_from_main_application_to_plugin_renderer->container.screen_y_offset = screen_y_offset;
        if(p_dlt_ctx)
        {
            DLT_LOG(*p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("mlink_wl_initialize: context_from_main_application_to_plugin_renderer created to address"),
                    DLT_PTR(context_from_main_application_to_plugin_renderer)
                    );
        }
        initDetail.cdfApplicationContext = context_from_main_application_to_plugin_renderer;

        const VNCViewerError error = pViewerSdk->vncViewerLoadRendererFromFile(
                pViewer,
                "vcdfcl",
                (VNCRendererViewerContext)&initDetail);
        if (error == VNCViewerErrorNone)
        {
            if(p_dlt_ctx)
            {
                DLT_LOG(*p_dlt_ctx,DLT_LOG_INFO,
                        DLT_STRING("mlink_wl_initialize: Plugin Renderer successfully loaded"));
            }
        }
        else
        {
            if(p_dlt_ctx)
            {
                DLT_LOG(*p_dlt_ctx,DLT_LOG_INFO,
                        DLT_STRING("mlink_wl_initialize: Plugin Renderer loading failed"));
            }
            is_h264_enabled = 0;
        }
    }
    else
    {
        context_from_main_application_to_plugin_renderer = NULL;
    }


    if( !pViewerSdk || !pViewer ||  !p_framebuffer_pixelformat || !pErrorCb)
    {
        if(p_dlt_ctx)
        {
            DLT_LOG(*p_dlt_ctx,DLT_LOG_ERROR,
                    DLT_STRING("mlink_wl_initialize: incomplete parameter list"));
        }
        return NULL;
    }

    if(!g_initialized_rwlock)
    {
        int rc = pthread_rwlock_init(&g_rwlock, NULL);
        if (rc)
        {
            if(p_dlt_ctx)
            {
                DLT_LOG(*p_dlt_ctx,DLT_LOG_ERROR,
                        DLT_STRING("mlink_wl_initialize: Failed to init rwlock with rc = "), DLT_INT32(rc));
            }
            return NULL;
        }
        else
        {
            g_initialized_rwlock = 1;
        }
    }

    pthread_rwlock_wrlock(&g_rwlock);
    mlink_wl_context * p_wl_ctx = malloc(sizeof(mlink_wl_context));

    if(p_dlt_ctx && context_from_main_application_to_plugin_renderer != NULL)
    {
        context_from_main_application_to_plugin_renderer->container.p_wayland = p_wl_ctx->p_wl_display;
        DLT_LOG(*p_dlt_ctx,DLT_LOG_INFO,
                DLT_STRING("mlink_wl_initialize: p_wl_ctx created at address"), DLT_PTR(p_wl_ctx),
                DLT_STRING("and container.p_wayland points to"), DLT_PTR(context_from_main_application_to_plugin_renderer->container.p_wayland));
    }

    if (p_wl_ctx)
    {
        memset(p_wl_ctx, 0, sizeof(mlink_wl_context));

        if(!mlink_wl_if_init(p_wl_ctx))
          {
            free(p_wl_ctx);
            p_wl_ctx = NULL;

            pthread_rwlock_unlock(&g_rwlock);
            return NULL;
          }

        p_wl_ctx->pErrorCb = pErrorCb;

        p_wl_ctx->p_dlt_ctx = p_dlt_ctx;
        p_wl_ctx->pViewer = pViewer;
        p_wl_ctx->pViewerSdk = pViewerSdk;
        p_wl_ctx->layer_id = layer_id;
        p_wl_ctx->surface_id = surface_id;
        p_wl_ctx->screen_x_offset = screen_x_offset;
        p_wl_ctx->screen_y_offset = screen_y_offset;
        p_wl_ctx->screen_width = screen_width;
        p_wl_ctx->screen_height = screen_height;
        p_wl_ctx->force_fullscreen = force_fullscreen;
        p_wl_ctx->flags = flags;
        p_wl_ctx->is_h264_enabled = is_h264_enabled;
        p_wl_ctx->p_wl_display = NULL;

        p_wl_ctx->bpp = p_framebuffer_pixelformat->bitsPerPixel;
        printf("pixel: bpp: %d, rgb: %d %d %d\n", p_framebuffer_pixelformat->bitsPerPixel,
            p_framebuffer_pixelformat->redMax,
            p_framebuffer_pixelformat->greenMax,
            p_framebuffer_pixelformat->blueMax);

        if(!mlink_wl_if_set_surface_format(p_wl_ctx, p_framebuffer_pixelformat))
          {
            if(p_dlt_ctx)
            {
                DLT_LOG(*p_dlt_ctx,DLT_LOG_ERROR,
                        DLT_STRING("mlink_wl_initialize: unsupported pixelformat"));
            }
            /* Free the memory allocated in mlink_if_init */
            mlink_free_mem((void **)&p_wl_ctx->p_wl_if_context);

            mlink_free_mem((void **)p_wl_ctx);

            pthread_rwlock_unlock(&g_rwlock);
            return NULL;
          }

        if (!mlink_wl_create_wl_context(p_wl_ctx))
          {
            if(p_dlt_ctx)
            {
                DLT_LOG(*p_dlt_ctx,DLT_LOG_ERROR,
                        DLT_STRING("mlink_wl_initialize: Unable to create Wayland Context"));
            }
            /* Free the memory allocated in mlink_if_init */
            mlink_free_mem((void **)&p_wl_ctx->p_wl_if_context);

            mlink_free_mem((void **)p_wl_ctx);

            pthread_rwlock_unlock(&g_rwlock);
            return NULL;
          }
    }

    pthread_rwlock_unlock(&g_rwlock);
    return p_wl_ctx;
}

void mlink_wl_enable_multitouch(mlink_wl_context * p_wl_ctx, vnc_int32_t enable)
{
  if(p_wl_ctx)
    {
      p_wl_ctx->support_touch = enable;
    }
}

void mlink_wl_finalize(mlink_wl_context * p_ctx)
{
    if(p_ctx)
    {
        if(g_initialized_rwlock)
        {
            pthread_rwlock_wrlock(&g_rwlock);
        }
        else
        {
            if(p_ctx->p_dlt_ctx)
            {
                DLT_LOG(*p_ctx->p_dlt_ctx,DLT_LOG_ERROR,
                         DLT_STRING("mlink_wl_finalize: rwlock is not initialized"));
            }
            return;
        }

        /* The below API may be called without initializtion
           So all the NULL conditions should be properly checked */

        if (p_ctx->p_dlt_ctx)
        {
            DLT_LOG(*p_ctx->p_dlt_ctx,DLT_LOG_INFO,
                    DLT_STRING("mlink_wl_finalize"));
        }

        if (p_ctx->is_h264_enabled)
        {
            context_from_main_application_to_plugin_renderer->container.is_streaming_stopped = 1;
            pthread_join(waitWayland, NULL);
        }

        mlink_wl_free_image_buffer(p_ctx);

        mlink_wl_if_deinit(p_ctx);
        mlink_wl_destroy_wl_context(p_ctx);

        free(p_ctx);
        p_ctx = NULL;
        free(context_from_main_application_to_plugin_renderer);
        context_from_main_application_to_plugin_renderer = NULL;

        pthread_rwlock_unlock(&g_rwlock);
    }
}

void mlink_wl_server_init_callback(VNCViewer *pViewer, void *vpContext,
        vnc_uint16_t width, vnc_uint16_t height, const char *desktopName,
        const VNCPixelFormat *pServerNativePixelFormat)
{
    pViewer = pViewer;
    desktopName = desktopName;
    pServerNativePixelFormat = pServerNativePixelFormat;

    mlink_dlt_context * p_dlt_ctx = ((mlink_adapter_context *)vpContext)->dlt_ctx;
    if(p_dlt_ctx)
    {
        DltContext * p_dlt = mlink_dlt_get_dlt_context(p_dlt_ctx);
        DLT_LOG(*p_dlt,DLT_LOG_INFO,
                DLT_STRING("mlink_wl:  server_init_cb: "), DLT_UINT16(width), DLT_UINT16(height));
    }

    mlink_wl_context * p_wl_ctx = ((mlink_adapter_context *) vpContext)->wl_ctx;
    if(p_wl_ctx)
    {
        if (!p_wl_ctx->is_h264_enabled)
        {
            if (!mlink_wl_create_image_buffer(p_wl_ctx, width, height))
            {
                if (p_wl_ctx->pErrorCb)
                {
                    p_wl_ctx->pErrorCb((mlink_adapter_context *) vpContext,
                            "server_init_callback: could not create image buffer");
                }
            }
        }
        else
        {
            if (p_wl_ctx->p_wl_display != NULL)
            {
                p_wl_ctx->intialized = 1;
            }
        }
    }

}


void
mlink_wl_desktop_resize_callback(VNCViewer *pViewer, void *vpContext,
    vnc_uint16_t newWidth, vnc_uint16_t newHeight)
{
  pViewer = pViewer;
  mlink_wl_context * p_wl_ctx = ((mlink_adapter_context *) vpContext)->wl_ctx;
  if (p_wl_ctx)
    {
      if (p_wl_ctx->intialized)
        {
          if (p_wl_ctx->p_dlt_ctx)
            {
              DLT_LOG(*p_wl_ctx->p_dlt_ctx, DLT_LOG_INFO,
                  DLT_STRING("mlink_wl: desktop_resize_cb"),
                  DLT_UINT32(newWidth), DLT_UINT32(newHeight));
            }
          mlink_wl_free_image_buffer(p_wl_ctx);
          if (!mlink_wl_create_image_buffer(p_wl_ctx, newWidth, newHeight))
            {
              p_wl_ctx->pErrorCb((mlink_adapter_context *) vpContext,
                  "desktop_resize_callback: could not initialize layermanager");
            }
        }
    }
}

vnc_uint8_t *
mlink_wl_lock_rectangle_ex_callback(VNCViewer *pViewer, void *vpContext,
        const VNCRectangle *pRectangle, size_t *pStride,
        VNCViewerLockRectangleReason reason)
{
    pViewer = pViewer;
    pRectangle = pRectangle;
    pStride = pStride;
    reason = reason;

    vnc_uint8_t * ptr = NULL;

    mlink_wl_context * p_wl_ctx = ((mlink_adapter_context *) vpContext)->wl_ctx;
    if (p_wl_ctx)
    {
        if (p_wl_ctx->intialized == 1)
        {
            *pStride = p_wl_ctx->stride;

            ptr = p_wl_ctx->p_image[0] + pRectangle->topLeft.x * p_wl_ctx->bpp / 8
                    + pRectangle->topLeft.y * p_wl_ctx->stride;
        }
    }

    return ptr;
}


void mlink_wl_framebuffer_update_end_callback(VNCViewer *pViewer,
        void *vpContext, const VNCRectangle *pInvalidatedRectangles,
        size_t rectangleCount)
{
    pViewer = pViewer;
    pInvalidatedRectangles = pInvalidatedRectangles;
    rectangleCount = rectangleCount;

    mlink_wl_context * p_wl_ctx = ((mlink_adapter_context *) vpContext)->wl_ctx;
    if (p_wl_ctx)
    {
        if (!p_wl_ctx->is_h264_enabled && p_wl_ctx->intialized)
        {
            mlink_wl_if_draw(p_wl_ctx);

// #define DEBUG_IMG
#ifdef  DEBUG_IMG
            static int i = 0;
            char filename[40];
            sprintf(filename,"/tmp/img_%06d.rgb",i);
            FILE * fd = fopen(filename,"wb");
            if(fd)
            {
                printf("writing img: %d x %d = %d bytes \n",
                        p_wl_ctx->stride, p_wl_ctx->height,p_wl_ctx->stride * p_wl_ctx->height);
                int rv  = fwrite((void*)p_wl_ctx->p_image[0], p_wl_ctx->stride, p_wl_ctx->height, fd);
                printf("return code: %d \n", rv);
                i++;
                fclose(fd);
            }
#endif
        }
    }
}


int
mlink_wl_is_equal_vnc_pixelformat(VNCPixelFormat * p_1, VNCPixelFormat * p_2)
{
  int ret = 0;
  if( p_1->bitsPerPixel  == p_2->bitsPerPixel  &&
      p_1->depth         == p_2->depth         &&
      p_1->bigEndianFlag == p_2->bigEndianFlag &&
      p_1->trueColorFlag == p_2->trueColorFlag &&
      p_1->redMax        == p_2->redMax        &&
      p_1->greenMax      == p_2->greenMax      &&
      p_1->blueMax       == p_2->blueMax       &&
      p_1->redShift      == p_2->redShift      &&
      p_1->greenShift    == p_2->greenShift    &&
      p_1->blueShift     == p_2->blueShift     )
    {
      ret=1;
    }

  return ret;
}


void
mlink_wl_pump_events(mlink_wl_context * p_wl_ctx)
{
  if (p_wl_ctx)
  {
      if(g_initialized_rwlock)
      {
          int rc = pthread_rwlock_rdlock(&g_rwlock);
          if (rc)
          {
              if(p_wl_ctx->p_dlt_ctx)
              {
                  DLT_LOG(*p_wl_ctx->p_dlt_ctx, DLT_LOG_ERROR,
                          DLT_STRING("mlink_wl_pump_events: rwlock applies a read lock failed with rc = "), DLT_INT32(rc));
              }
          }
      }
      else
      {
          return;
      }

      if (p_wl_ctx->intialized)
      {
          if ((p_wl_ctx->is_h264_enabled)
              && (context_from_main_application_to_plugin_renderer->container.is_streaming_stopped == 1))
          {
             pthread_rwlock_unlock(&g_rwlock);
             return;
          }
          struct pollfd pfd[1];
          pfd[0].fd = wl_display_get_fd(p_wl_ctx->p_wl_display);
          pfd[0].events = POLLIN;

          //Mark this thread as a reading thread
          while (0
              != wl_display_prepare_read_queue(p_wl_ctx->p_wl_display,
                  p_wl_ctx->p_wl_input_queue))
            {
              if(p_wl_ctx->p_dlt_ctx)
              {
                  DLT_LOG(*p_wl_ctx->p_dlt_ctx,DLT_LOG_INFO,
                          DLT_STRING("mlink_wl: wl_display_prepare_read_queue() failed"));
              }

              wl_display_dispatch_queue_pending(p_wl_ctx->p_wl_display,
                  p_wl_ctx->p_wl_input_queue);
            }

          //Flush events before going to blocking wait
          wl_display_flush(p_wl_ctx->p_wl_display);

          // Check for Wayland event (not necessarily input event)
          int retval = poll(pfd, 1, 0);

          if(retval <= 0)
            {
              // cancel in read if no event is pending
              wl_display_cancel_read(p_wl_ctx->p_wl_display);
            }
          else
            {
              wl_display_read_events(p_wl_ctx->p_wl_display);
              wl_display_dispatch_queue_pending(p_wl_ctx->p_wl_display,
                  p_wl_ctx->p_wl_input_queue);
            }
        }

        pthread_rwlock_unlock(&g_rwlock);
    }
}

/*lint +e160 +e505 +e578*/
/*lint +e40 +e413 +e613*/
/*lint +e429*/
